import xbmc
from urllib.request import Request, urlopen
from urllib.parse import quote
from urllib.error import URLError
import time
import json

# Volume map: Maps /Volumes/X/ to the server's expected path
VOLUME_MAP = {
    "3": "/mnt/sda1/Movies3/",
    "4": "/mnt/sda1/Movies4/",
    "5": "/mnt/sda1/Movies5/",
    "7": "/mnt/sda1/Movies7/",
    "8": "/mnt/sda1/Movies8/"
}

# Sony Bravia TV configuration
BRAVIA_IP = "192.168.1.55"
BRAVIA_PSK = "0000"  # Confirmed working

# Oppo device HTTP configuration
OPPO_IP = "192.168.1.57"
OPPO_PORT = 436

def switch_to_hdmi4():
    url = f"http://{BRAVIA_IP}/sony/avContent"
    payload = '{"method":"setPlayContent","params":[{"uri":"extInput:hdmi?port=4"}],"id":1,"version":"1.0"}'
    headers = {"Content-Type": "application/json", "X-Auth-PSK": BRAVIA_PSK}
    
    try:
        req = Request(url, data=payload.encode('utf-8'), headers=headers, method="POST")
        xbmc.log(f"Sending HDMI switch request to Bravia TV: {url}", xbmc.LOGINFO)
        with urlopen(req) as response:
            status_code = response.getcode()
            response_content = response.read().decode('utf-8', errors='ignore')
            xbmc.log(f"Bravia HDMI switch response: status code {status_code}, content: {response_content}", xbmc.LOGINFO)
    except URLError as e:
        xbmc.log(f"Failed to switch HDMI input on Bravia TV: {e}", xbmc.LOGERROR)

def switch_to_hdmi1():  # Switch back to HDMI 1 (adjust port as needed)
    url = f"http://{BRAVIA_IP}/sony/avContent"
    payload = '{"method":"setPlayContent","params":[{"uri":"extInput:hdmi?port=1"}],"id":1,"version":"1.0"}'
    headers = {"Content-Type": "application/json", "X-Auth-PSK": BRAVIA_PSK}
    
    try:
        req = Request(url, data=payload.encode('utf-8'), headers=headers, method="POST")
        xbmc.log(f"Sending HDMI switch request to Bravia TV (HDMI 1): {url}", xbmc.LOGINFO)
        with urlopen(req) as response:
            status_code = response.getcode()
            response_content = response.read().decode('utf-8', errors='ignore')
            xbmc.log(f"Bravia HDMI switch response: status code {status_code}, content: {response_content}", xbmc.LOGINFO)
    except URLError as e:
        xbmc.log(f"Failed to switch HDMI input on Bravia TV: {e}", xbmc.LOGERROR)

def check_oppo_playback_status():
    url = f"http://{OPPO_IP}:{OPPO_PORT}/getmovieplayinfo"
    try:
        req = Request(url)
        with urlopen(req) as response:
            status_code = response.getcode()
            response_content = response.read().decode('utf-8', errors='ignore')
            xbmc.log(f"Oppo HTTP status response: status code: {status_code}, content: {response_content}", xbmc.LOGINFO)
            return response_content
    except URLError as e:
        xbmc.log(f"HTTP status request failed: {e}", xbmc.LOGERROR)
        return None

def monitor_oppo_playback():
    xbmc.log("Starting Oppo playback monitoring", xbmc.LOGINFO)
    
    # Wait for Oppo to start playing, with a timeout
    start_time = time.time()
    timeout = 10  # 10 seconds to allow playback to begin
    while time.time() - start_time < timeout:
        response = check_oppo_playback_status()
        if response:
            try:
                status_data = json.loads(response)
                success = status_data.get("success", True)
                if success and status_data.get("playinfo", {}).get("e_play_status", -1) == 0:
                    xbmc.log("Oppo confirmed playing, entering monitoring loop", xbmc.LOGINFO)
                    break
            except json.JSONDecodeError:
                xbmc.log("Failed to parse Oppo status response as JSON", xbmc.LOGWARNING)
        xbmc.log("Waiting for Oppo to start playback", xbmc.LOGINFO)
        time.sleep(1)
    else:
        xbmc.log("Oppo failed to start playback within timeout", xbmc.LOGERROR)
        switch_to_hdmi1()
        return

    # Main monitoring loop
    while True:
        response = check_oppo_playback_status()
        if response:
            try:
                status_data = json.loads(response)
                success = status_data.get("success", True)
                if success:
                    play_status = status_data.get("playinfo", {}).get("e_play_status", -1)
                    if play_status == 0:
                        xbmc.log("Oppo is playing", xbmc.LOGINFO)
                    elif play_status == 56:
                        xbmc.log("Oppo is paused", xbmc.LOGINFO)
                    else:
                        xbmc.log(f"Oppo in unknown play state: e_play_status={play_status}", xbmc.LOGWARNING)
                elif not success and status_data.get("errcode1", 0) == -5:
                    xbmc.log("Detected playback stopped on Oppo", xbmc.LOGINFO)
                    switch_to_hdmi1()  # Switch back to HDMI 1
                    break
            except json.JSONDecodeError:
                xbmc.log("Failed to parse Oppo status response as JSON", xbmc.LOGWARNING)
        else:
            xbmc.log("No valid response from Oppo", xbmc.LOGWARNING)
        
        time.sleep(7)
        if xbmc.Monitor().abortRequested():
            xbmc.log("Kodi abort requested, stopping monitoring", xbmc.LOGINFO)
            break

class MyPlayer(xbmc.Player):
    def __init__(self):
        xbmc.Player.__init__(self)
        xbmc.log("MyPlayer initialized", xbmc.LOGINFO)

    def onPlayBackStarted(self):
        xbmc.log("Playback started detected", xbmc.LOGINFO)
        
        xbmc.sleep(500)
        
        try:
            file_path = self.getPlayingFile()
            xbmc.log(f"Original file path: {file_path}", xbmc.LOGINFO)
            
            normalized_path = file_path
            if file_path.startswith("/Volumes/"):
                volume_num = file_path.split("/")[2]
                if volume_num in VOLUME_MAP:
                    normalized_path = file_path.replace(f"/Volumes/{volume_num}/", VOLUME_MAP[volume_num])
                    xbmc.log(f"Normalized file path: {normalized_path}", xbmc.LOGINFO)
                else:
                    xbmc.log(f"Volume {volume_num} not found in VOLUME_MAP, skipping request", xbmc.LOGWARNING)
                    return
            else:
                xbmc.log("File path not under /Volumes/, using original path", xbmc.LOGINFO)
        except Exception as e:
            xbmc.log(f"Error getting or normalizing file: {e}", xbmc.LOGERROR)
            return

        switch_to_hdmi4()

        json_payload = f'{{"path":"{normalized_path}","index":0,"type":1,"appDeviceType":2,"extraNetPath":"192.168.1.238/Storage/UHD/","playMode":0}}'
        encoded_payload = quote(json_payload)
        url = f"http://{OPPO_IP}:{OPPO_PORT}/playnormalfile?{encoded_payload}"
        xbmc.log(f"Constructed URL: {url}", xbmc.LOGINFO)

        for attempt in range(2):
            try:
                req = Request(url)
                xbmc.log(f"Sending HTTP request (attempt {attempt + 1})", xbmc.LOGINFO)
                with urlopen(req) as response:
                    status_code = response.getcode()
                    response_content = response.read().decode('utf-8', errors='ignore')
                    xbmc.log(f"HTTP response: status code: {status_code}, content: {response_content}", xbmc.LOGINFO)
                    
                    if '"success":true' in response_content:
                        xbmc.log("Server confirmed successful playback", xbmc.LOGINFO)
                        break
                    else:
                        xbmc.log("Server rejected request (success:false)", xbmc.LOGWARNING)
                        if attempt == 0:
                            xbmc.log("Retrying after 2-second delay", xbmc.LOGINFO)
                            xbmc.sleep(2000)
                        else:
                            xbmc.log("Final retry failed", xbmc.LOGERROR)
            except URLError as e:
                xbmc.log(f"HTTP request failed: {e}", xbmc.LOGERROR)
                if attempt == 0:
                    xbmc.log("Retrying after 1-second delay", xbmc.LOGINFO)
                    xbmc.sleep(1000)
                else:
                    xbmc.log("Final retry failed due to network error", xbmc.LOGERROR)

        self.stop()
        xbmc.executebuiltin("Action(Stop)")
        monitor_oppo_playback()

class MyMonitor(xbmc.Monitor):
    def __init__(self, player):
        self.player = player
        xbmc.log("MyMonitor initialized", xbmc.LOGINFO)

    def onAbortRequested(self):
        xbmc.log("Abort requested", xbmc.LOGINFO)

player = MyPlayer()
monitor = MyMonitor(player)

xbmc.log("Script started, entering main loop", xbmc.LOGINFO)
while not monitor.abortRequested():
    if player.isPlaying():
        xbmc.log("Player is active", xbmc.LOGINFO)
        xbmc.sleep(1000)
    else:
        xbmc.log("Player is idle", xbmc.LOGDEBUG)
        monitor.waitForAbort(1)